import datetime
import sqlite3

import openpyxl
import openpyxl.styles

from simple_term_menu import TerminalMenu

from buchung import Buchung, loadBuchungen
from person import loadPersonen, Person
from room import loadRaeume, Room


def login():
    menu = TerminalMenu(["Mitarbeiter", "Verwaltung"], title="Wie möchten Sie sich anmelden?")
    auswahl = menu.show()

    if auswahl == 0:  # Mitarbeiter
        mitarbeiterMenu()
    elif auswahl == 1:  # Verwaltung
        verwaltungMenu()


def mitarbeiterMenu():
    menu = TerminalMenu(["Buchungen für Tag einsehen", "Raum Buchen", "Abmelden"],
                        title="Welche Aktion wollen Sie ausführen? (Mitarbeiter)")
    auswahl = menu.show()

    if auswahl == 0:
        buchungEinsehen(mitarbeiterMenu)
    elif auswahl == 1:
        raumBuchen(mitarbeiterMenu)
    elif auswahl == 2:
        login()


def verwaltungMenu():
    menu = TerminalMenu(
        ["Buchungen für Tag einsehen", "Buchungsarchiv einsehen", "Raum Buchen", "Buchungen ändern",
         "Mitarbeiter verwalten", "Räume verwalten", "Archiv als Excel-Datei exportieren", "Abmelden"],
        title="Welche Aktion wollen Sie ausführen? (Verwaltung)")
    auswahl = menu.show()

    if auswahl == 0:
        buchungEinsehen(verwaltungMenu)
    elif auswahl == 1:
        archivEinsehen(verwaltungMenu)
    elif auswahl == 2:
        raumBuchen(verwaltungMenu)
    elif auswahl == 3:
        buchungenAendern(verwaltungMenu)
    elif auswahl == 4:
        mitarbeiterVerwalten(verwaltungMenu)
    elif auswahl == 5:
        raeumeBearbeiten(verwaltungMenu)
    elif auswahl == 6:
        excelExport(verwaltungMenu)
    elif auswahl == 7:
        login()


def buchungenAendern(returnTo=login):
    global db

    buchungen = loadBuchungen(db, "date(date) = date('now')")

    buchungenStrings = [f"{buchung.timeSlot}: {buchung.person.name} in {buchung.room.name}" for buchung in buchungen]

    menu = TerminalMenu(buchungenStrings, title="Welche Buchung wollen Sie ändern?")
    auswahl = menu.show()

    if auswahl is None:
        print("Keine Buchung Ausgewählt")
        returnTo()
        return

    buchungAendern(buchungen[auswahl])

    returnTo()


def buchungAendern(buchung: Buchung):
    global db

    menu = TerminalMenu(["Person ändern", "Raum ändern", "Buchung löschen"],
                        title=f"Welche Aktion wollen sie auf der Buchung {buchung.timeSlot} von {buchung.person.name} in {buchung.room.name}")
    auswahl = menu.show()

    match auswahl:
        case 0:  # Person ändern
            personen = loadPersonen(db)
            menuP = TerminalMenu([person.name for person in personen])
            auswahlP = menuP.show()

            buchung.person = personen[auswahlP]
            buchung.updateDB(db)

        case 1:  # Raum ändern
            raeume = loadRaeume(db)
            menuR = TerminalMenu([raum.name for raum in raeume])
            auswahlR = menuR.show()

            buchung.room = raeume[auswahlR]
            buchung.updateDB(db)

        case 2:  # Buchung löschen
            buchung.removeFromDB(db)


def buchungEinsehen(returnTo=login):
    global db

    # buchungen für den Tag laden
    buchungen = loadBuchungen(db, "date(date) = date('now')")

    # clear term
    print("\033c", end="")

    if len(buchungen) == 0:
        print("Es gibt keine Buchungen für heute.")
        returnTo()

    # get the max name and room name length so that we can align them
    maxNameLen = max(map(lambda x: len(x.person.name), buchungen))
    maxRaumLen = max(map(lambda x: len(x.room.name), buchungen))

    for buchung in buchungen:
        print(f"{buchung.person.name.ljust(maxNameLen)} hat den Raum {buchung.room.name.ljust(maxRaumLen)} um {buchung.timeSlot} gebucht. ")

    returnTo()


def archivEinsehen(returnTo=login):
    global db

    buchungen = loadBuchungen(db)

    # clear the terminal
    print("\033c", end="")

    if len(buchungen) == 0:
        print("Es gibt keine Buchungen.")
        returnTo()

    maxNameLen = max(map(lambda x: len(x.person.name), buchungen))
    maxRaumLen = max(map(lambda x: len(x.room.name), buchungen))

    lastDate = None
    for buchung in buchungen:
        if lastDate != buchung.date:
            # bold formatting
            print(f"\033[1m{buchung.date.isoformat()}\033[0m: ")
            lastDate = buchung.date
        print(
            f"{buchung.person.name.ljust(maxNameLen)} hat den Raum {buchung.room.name.ljust(maxRaumLen)} um {buchung.timeSlot} gebucht. ")

    returnTo()


def excelExport(returnTo=login):
    global db

    buchungen = loadBuchungen(db)

    if len(buchungen) == 0:
        print("Es gibt keine Buchungen.")
        returnTo()

    workbook = openpyxl.Workbook()

    currentSheet = None
    lastDate = None
    i = 1

    workbook.remove(workbook.active)
    for buchung in buchungen:
        if lastDate != buchung.date:
            currentSheet = workbook.create_sheet(buchung.date.isoformat())

            # header row
            currentSheet["A1"] = "Slot"
            currentSheet["A1"].font = openpyxl.styles.Font(bold=True)
            currentSheet["B1"] = "Raum"
            currentSheet["B1"].font = openpyxl.styles.Font(bold=True)
            currentSheet["C1"] = "Person"
            currentSheet["C1"].font = openpyxl.styles.Font(bold=True)

            # in der 1. (zweiten) Reihe anfangen
            i = 1
            lastDate = buchung.date
        # i+1 da excel 1 indexed ist
        currentSheet[f"A{i+1}"] = buchung.timeSlot
        currentSheet[f"B{i+1}"] = buchung.room.name
        currentSheet[f"C{i+1}"] = buchung.person.name
        i += 1

    workbook.save("./export.xlsx")
    returnTo()


def raumBuchen(returnTo=login):
    global db

    # Show and select person
    personen = loadPersonen(db)
    menu = TerminalMenu(map(lambda x: x.name, personen), title="Welche Person möchte einen Raum buchen?")
    auswahl = menu.show()
    person = personen[auswahl]

    # Show and select room
    raeume = loadRaeume(db)
    menu = TerminalMenu(map(lambda x: x.name, raeume), title="Welchen Raum wollen Sie buchen?")
    auswahl = menu.show()
    raum = raeume[auswahl]

    buchungen = loadBuchungen(db, f"date(date) = date('now') AND room = {raum.id}")

    # create the timeSlot array and remove slots were the room is already booked
    timeSlots = []
    for i in range(8, 18):
        if not f"{i:02d}:00" in map(lambda x: x.timeSlot, buchungen):
            timeSlots.append(f"{i:02d}:00")
        else:
            timeSlots.append(None)  # Fügt einen leeren, nicht auswählbaren Platz hinzu.

        if not f"{i:02d}:30" in map(lambda x: x.timeSlot, buchungen):
            timeSlots.append(f"{i:02d}:30")
        else:
            timeSlots.append(None)
    menu = TerminalMenu(timeSlots, title="Welche Zeit(en) wollen Sie buchen?", multi_select=True,
                        show_multi_select_hint=True)
    auswahl = menu.show()
    # remove crashes
    auswahl = [] if auswahl is None else auswahl
    for i in auswahl:
        timeSlot = timeSlots[i]

        buchung = Buchung(raum, person, timeSlot, datetime.date.today())
        buchung.updateDB(db)

    returnTo()


def mitarbeiterVerwalten(returnTo=login):
    # Show and select person
    personen = loadPersonen(db)
    personen.append("new")
    menu = TerminalMenu(map(lambda x: "Neue Person hinzufügen" if x == "new" else x.name, personen),
                        title="Welche Person möchten Sie modifizieren?")
    auswahl = menu.show()
    person = personen[auswahl]

    if person == "new":
        mitarbeiterHinzu()
        returnTo()
    else:
        mitarbeiterBearbeiten(person)
        returnTo()


def mitarbeiterBearbeiten(mitarbeiter: Person):
    global db

    menu = TerminalMenu(["Name ändern", "Person löschen"],
                        title=f"Welche Aktion wollen sie an der Person {mitarbeiter.name} ausführen? ")
    auswahl = menu.show()

    match auswahl:
        case 0:  # Name ändern
            name = input("Name: ")

            mitarbeiter.name = name
            mitarbeiter.updateDB(db)

        case 1:  # Person löschen
            mitarbeiter.removeFromDB(db)


def mitarbeiterHinzu():
    global db

    name = input("Name: ")

    p = Person(name)
    p.updateDB(db)


def raumHinzu():
    global db

    name = input("Name: ")

    r = Room(name)
    r.updateDB(db)


def raeumeBearbeiten(returnTo=login):
    # Show and select raum
    raeume = loadRaeume(db)
    raeume.append("new")
    menu = TerminalMenu(map(lambda x: "Neuen Raum hinzufügen" if x == "new" else x.name, raeume),
                        title="Welchen Raum möchten Sie modifizieren?")
    auswahl = menu.show()
    raum = raeume[auswahl]

    if raum == "new":
        raumHinzu()
        returnTo()
    else:
        raumBearbeiten(raum)
        returnTo()


def raumBearbeiten(raum: Room):
    global db

    menu = TerminalMenu(["Name ändern", "Raum löschen"],
                        title=f"Welche Aktion wollen sie an dem Raum {raum.name} ausführen? ")
    auswahl = menu.show()

    match auswahl:
        case 0:  # Name ändern
            name = input("Name: ")

            raum.name = name
            raum.updateDB(db)

        case 1:  # Raum löschen
            raum.removeFromDB(db)


if __name__ == "__main__":
    db = sqlite3.connect("db.sqlite")

    login()
