import datetime
import sqlite3

from room import Room
from person import Person


class Buchung:
    def __init__(self, room: Room, person: Person, timeSlot: str, date: datetime.date, id=None):
        self.room = room
        self.person = person
        self.timeSlot = timeSlot
        self.date = date
        self.id = id

    def updateDB(self, db: sqlite3.Connection):
        # updates the db-entry if it exists else insert into the db
        cur = db.cursor()
        if self.id is None:  # Not inserted into table yet
            cur.execute("INSERT INTO slot (room, person, timeSlot, date) VALUES (?, ?, ?, date(?))",
                        [self.room.id, self.person.id, self.timeSlot, self.date])
        else:
            cur.execute("UPDATE slot SET room=?, person=?, timeSlot=?, date=date(?) WHERE id=?",
                        [self.room.id, self.person.id, self.timeSlot, self.date, self.id])

        db.commit()

    def removeFromDB(self, db: sqlite3.Connection):
        cur = db.cursor()
        if self.id is None:
            print("Not in db")
        else:
            cur.execute("DELETE FROM slot WHERE id=?", [self.id])
            db.commit()

    def __repr__(self):
        return f"(Raum={self.room}, Person={self.person}, timeSlot={self.timeSlot}, date={self.date}, id={self.id})"


def loadBuchungen(db: sqlite3.Connection, condition="true") -> list[Buchung]:
    # loads every buchung as a class
    # condition is a sql condition to be used when you for example want to only get them for one day
    cur = db.cursor()
    cur.execute("SELECT slot.id, r.name, r.id, p.name, p.id, timeSlot, date(date) FROM slot "
                "INNER JOIN person p on p.id = slot.person "
                "INNER JOIN room r on r.id = slot.room "
                f"WHERE {condition} "
                "ORDER BY slot.date, slot.room, slot.timeSlot ")
    buchungen = cur.fetchall()

    out = []

    for buchung in buchungen:
        id = buchung[0]
        room = Room(buchung[1], buchung[2])
        person = Person(buchung[3], buchung[4])
        timeSlot = buchung[5]
        date = datetime.date.fromisoformat(buchung[6])

        out.append(Buchung(room, person, timeSlot, date, id))

    return out
